/*
 *  This script is designed to generate toy data that resemble molecular dynamics within a defined space (eg a cell)
 *  The entire image is the 'cell', and changes in pixel value intensities (0-255) represent molecular dynamics.
 *  To tune the properties of the dynamics, one can set the amount of space in the cell that can be dynamic (DYNAMIC),
 *  the exact or average duration of pixel dynamics (LIFETIME), the overall signal to noise ratio (NOISE & SIGNAL),
 *  and whether pixels are simply off and on or whether there are fluctuations in intensity (SIGNALSLOPE).
 *  Version history at end.
 *  
 */


DYNAMIC=newArray(0.1, 0.5, 1); //The fraction of the field that can possibly change frame to frame. [0-1]
pON=newArray(0.0002, 0.001, 0.005, 0.025, 0.125); //probability of a pixel turning 'on'. [0-1]
LIFETIME=newArray(5, 10, 15, 20); // Lifetime means. [Integer>0]
POISSON=true; // Whether lifetimes should follow a Poisson distribution. [true, false]
SPREAD=1; // [0,1]. If 0, lifetime distribution is exactly the lifetime only. If 1, flat distribtion from 1 to lifetime*2-1
NOISE=91; //noise constant, 0-255. Gets multiplied by a random number to set background pixel value. 0=no noise.
SIGNAL=255; //SIGNAL constant. Sets peak value of 'on' pixels
SIGNALSLOPE=0; //Constant value by which signal increases or decreases per frame when 'on'. Meaningful values will range from 0 to 510/LIFETIME
GAP=true; // Whether the gap between events should be deterministic (true) or probabilistic (false) THIS ISN'T USED YET
imgw=64; // image width
imgh=64; // image height
PATCHSIZE=4; // Size of patch edge
FRAMES=100; // The number of frames in each 'movie'

/*
 * If you want to do single values of any parameter, uncomment the line below and set the value as desired.
 */


N=1; // number of replicates per image. one is pretty much sufficient, very low variability between instances.
DYNAMIC=newArray(1); DYNAMIC[0]=0.6; //The fraction of the field that can possibly change frame to frame
//pON=newArray(1); pON[0]=0.1;  //The probability of a pixel changing
//LIFETIME=newArray(1); LIFETIME[0]=12; //The probability of a pixel changing

path=getDirectory("choose a folder to store images in");
setBatchMode(true);

for(dyn=0; dyn<DYNAMIC.length; dyn++)
	{
	for(pon=0; pon<pON.length; pon++)
		{
		for(cold=0; cold<1; cold++) // The idea of cold is to prevent pixels from turning back on
			{
			for(lt=0; lt<LIFETIME.length; lt++)
				{
				ps=buildDist(LIFETIME[lt]); // Build lifetime distribution using parameters set above
				for(n=0; n<N; n++)
					{
					// Initially make the substack overly long, so that we can 'initialize' it and stop before the end to 
					// avoid edge effects in time
					newImage("Dynamic-"+DYNAMIC[dyn]+"    pON-"+pON[pon]+ "    LIFETIME-"+LIFETIME[lt]+".tif", "8-bit black", imgw, imgh, FRAMES+5*LIFETIME[lt]);
					makeFakePAZ(DYNAMIC[dyn], pON[pon], LIFETIME[lt], cold);
					
					// Crop the toy data in time to the final FRAMES length, starting 3 lifetimes in.
					run("Make Substack...", "  slices="+3*LIFETIME[lt]+"-"+FRAMES+3*LIFETIME[lt]);
					
					saveAs(".tif", path+ "\\Dynamic-"+DYNAMIC[dyn]+"    pON-"+pON[pon]+ "    LIFETIME-"+LIFETIME[lt]+"   .tif");
					close("*");
					}
				}
			}
		}
	}

print("Values used to create this set of images:");
print("NOISE="+NOISE+"\nSIGNAL="+SIGNAL+"\nSIGNALSLOPE="+SIGNALSLOPE+"\nPOISSON="+POISSON+"\nSPREAD="+SPREAD);
print("Started at 4 lifetimes for each trace");
selectWindow("Log");
saveAs(".txt", path+"\\SettingToCreateImages.txt");

function makeFakePAZ(dynamic, pon, poff, cold)
	{
	/*
	 * The basic logic of building the toy data is:
	 * 1) Check whether a pixel is off
	 * 2) If off, decide whether it should turn on (random number < probability of turning on)
	 * 3) If turn on, set the pixel to the correct on value(s) for its entire lifetime
	 * 4) Repeat for all pixels in a frame, then move to next frame.
	 */
	
	run("Select All");
	Roi.getContainedPoints(xs, ys);
	getDimensions(width, height, channels, slices, frames);
	dynheight=dynamic*height;
	
	
	dynxs=Array.slice(xs, 0, floor(dynamic*xs.length));
	dynys=Array.slice(ys, 0, floor(dynamic*ys.length));
	bgxs=Array.slice(xs, floor(dynamic*xs.length)+1, xs.length);
	bgys=Array.slice(ys, floor(dynamic*xs.length)+1, xs.length);
	
	for(f=1; f<=slices; f++)
		{
		pop=newArray(0); // For use if cold==1; keeps track of on pixels to remove them from the active space
		for(row=0; row<dynheight; row+=PATCHSIZE)
			{
			for(col=0; col<width; col+=PATCHSIZE)
				{
				Stack.setSlice(f-1);
				pix=getValue(col, row);
				k=0;
				
				 
				if(pix<=NOISE) // If the pixel is off, check whether should turn on and if so set value and lifetime
					{
					rn=random;
					
					
					if(rn<pon) // Check whether pixel should turn on (if random < pON)
						{
						rnl=random; // Generate a random number to determine lifetime
						while(rnl>ps[k] && k<ps.length-1)
							k+=1;
	
						for(s=0; s<k+1; s++)
							{
							Stack.setSlice(f+s);
							setColor(SIGNAL-(SIGNALSLOPE*abs(k/2-s)));
							fillRect(col, row, PATCHSIZE, PATCHSIZE);
							}
	
						if(cold==1)
							{
							p1=newArray(1); p1[0]=i;
							pop=Array.concat(pop, p1);
							}	
						}
						else {
							for(pixx=0; pixx<PATCHSIZE; pixx++)
								{
								for(pixy=0; pixy<PATCHSIZE; pixy++)
									{
									bgr=random;
									setPixel(col+pixx, row+pixy, bgr*NOISE);
									}
								}
							
							}
					}
				}
			}
		
		if(cold==1)
			{
			for(p=0; p<pop.length; p++)
				{
				dynxs=Array.deleteIndex(dynxs, pop[p]-p); 
				dynys=Array.deleteIndex(dynys, pop[p]-p);
				}
			}

			
		Stack.setSlice(f);
		for(i=0; i<bgxs.length; i++)
			{
			bgr=random;
			setPixel(bgxs[i], bgys[i], bgr*NOISE);
			}
		
		}
	}


function buildDist(lifetime)
	{
	// Build lifetime distributions
	// lifetime can be any integer>0
	// If POISSON==true, the distribution will be a poisson distribution with mean=lifetime
	// If POISSON==false, the distribution will be either the single lifetime value itself (if SPREAD==0), or a flat distribution that ranges from 1 to 2*Lifetime-1

	if(POISSON==true)
		{
		k=0;
		fact=1;
		ps=newArray(1);
		totalp=0;
		while(totalp<0.99)
			{
			p=pow(lifetime, k)*pow(2.7182818, -lifetime)/fact;
			totalp+=p;
			if(k==0)
				{
				ps[0]=totalp;
				}
				else 
					{
					pn=newArray(1); pn[0]=totalp;
					ps=Array.concat(ps, pn);
					}
					
			k+=1;
			fact=fact*k;
			}
		} 
	else if(SPREAD==0)
		{
		// If lifetime distribution is exactly lifetime, make array of length lifetime with 100% probability at last index
		ps=newArray(lifetime);
		ps[lifetime-1]=1;
		}
	else {
			// For SPREAD=1, range will be 1 for 1, and 2*LT-1 for all others
			range=abs(2*lifetime-1);
			ps=newArray(range);
			for(p=0; p<range; p++)
				ps[p]=(p+1)/range;
			}

	return ps;
	}
